// $Id: CRegistry.cpp,v 1.2 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CRegistry.hpp"
using Exponent::Host::CRegistry;

//  ===========================================================================
bool CRegistry::readFromRegistry(const ERegistry registry, const CString &registryKey, const CString &key, CSystemString &input)
{
#ifdef WIN32
	char value[_MAX_PATH];
	HKEY netKey;
	DWORD lenKeyValue = _MAX_PATH;
	unsigned long dataType;

	if (registry == e_localMachine)
	{
		if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, registryKey.getString(), 0, KEY_READ, &netKey) == ERROR_SUCCESS)
		{
			if (RegQueryValueEx(netKey, key.getString(), 0, &dataType, (LPBYTE)&value, &lenKeyValue) == ERROR_SUCCESS)
			{
				input = value;
				RegCloseKey(netKey);
				return true;
			}
			RegCloseKey(netKey);
			return false;
		}
		else
		{
			return false;
		}
	}
	else if (registry == e_currentUser)
	{
		if(RegOpenKeyEx(HKEY_CURRENT_USER, registryKey.getString(), 0, KEY_READ, &netKey) == ERROR_SUCCESS)
		{
			if (RegQueryValueEx(netKey, key.getString(), 0, &dataType, (LPBYTE)&value, &lenKeyValue) == ERROR_SUCCESS)
			{
				input = value;
				RegCloseKey(netKey);
				return true;
			}
			RegCloseKey(netKey);
			return false;
		}
		else
		{
			return false;
		}
	}
	return false;
#else
	CSystemString theRegKey = registryKey;
	CSystemString theKey    = key;

	CFStringRef applicationID = theRegKey.getStringAsCFStringRef();
	CFStringRef cfKey     	  = theKey.getStringAsCFStringRef();
	CFStringRef cfValue	      = (CFStringRef)CFPreferencesCopyAppValue(cfKey, applicationID);

	if (applicationID)
	{
		CFRelease(applicationID);
	}

	if (cfKey)
	{
		CFRelease(cfKey);
	}

	if (cfValue)
	{
		input.setStringFromCFStringRef(cfValue);
		CFRelease(cfValue);
		return true;
	}
	else
	{
		return false;
	}
#endif
}

//  ===========================================================================
bool CRegistry::writeToRegistry(const ERegistry registry, const CString &registryKey, const CString &key, const CString &value)
{
#ifdef WIN32
	HKEY netKey;
	TCHAR szData[80];
	DWORD dwBufSize = 80;
	value.getString(szData, 80);

	if (registry == e_localMachine)
	{
		if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, registryKey.getString(), 0, KEY_SET_VALUE, &netKey) == ERROR_SUCCESS)
		{
			if (RegSetValueEx(netKey, key.getString(), 0, REG_SZ, (const BYTE*)&szData, dwBufSize) == ERROR_SUCCESS)
			{
				RegCloseKey(netKey);
				return true;
			}
			RegCloseKey(netKey);
			return false;
		}
		else
		{
			return false;
		}
	}
	if (registry == e_currentUser)
	{
		if(RegOpenKeyEx(HKEY_CURRENT_USER, registryKey.getString(), 0, KEY_SET_VALUE, &netKey) == ERROR_SUCCESS)
		{
			if (RegSetValueEx(netKey, key.getString(), 0, REG_SZ, (const BYTE*)&szData, dwBufSize) == ERROR_SUCCESS)
			{
				RegCloseKey(netKey);
				return true;
			}
			RegCloseKey(netKey);
			return false;
		}
		else
		{
			return false;
		}
	}

	return false;
#else
	CSystemString theRegKey = registryKey;
	CSystemString theKey    = key;
	CSystemString theValue  = value;

	// Of the form com.exponent.detlaspace
	CFStringRef applicationID = theRegKey.getStringAsCFStringRef();
	CFStringRef cfKey     	  = theKey.getStringAsCFStringRef();
	CFStringRef cfValue		  = theValue.getStringAsCFStringRef();
	bool complete			  = false;

	// Check that they are all valid
	if (applicationID && cfKey && cfValue)
	{
		// Store the value
		CFPreferencesSetAppValue(cfKey, cfValue, applicationID);
		
		// Synchronise
		complete = CFPreferencesAppSynchronize(applicationID);
	}

	if (applicationID)
	{
		CFRelease(applicationID);
	}

	if (cfKey)
	{
		CFRelease(cfKey);
	}

	if (cfValue)
	{
		CFRelease(cfValue);
	}

	return complete;
#endif
}

//  ===========================================================================
bool CRegistry::createRegistryEntry(const ERegistry registry, const CString &registryKey)
{
#ifdef WIN32
	HKEY netKey;

	if (registry == e_localMachine)
	{
		// Try to create the key
		if (RegCreateKeyEx(HKEY_LOCAL_MACHINE, registryKey.getString(), 0, NULL, REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &netKey, NULL) == ERROR_SUCCESS)
		{
			// Close the key
			RegCloseKey(netKey);
			return true;
		}
		return false;
	}
	if (registry == e_currentUser)
	{		
		// Try to create the key
		if (RegCreateKeyEx(HKEY_CURRENT_USER, registryKey.getString(), 0, NULL, REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &netKey, NULL) == ERROR_SUCCESS)
		{
			// Close the key
			RegCloseKey(netKey);
			return true;
		}
		return false;
	}

	return false;
#else
	// On macsx you cant create a plist entry like this..
	// HOwever one is created as soon as you try and write it :)
	return true;	
#endif	
}

//  ===========================================================================
bool CRegistry::deleteRegistryEntry(const ERegistry registry, const CString &registryKey, const CString &key)
{
#ifdef WIN32
	HKEY netKey;

	if (registry == e_localMachine)
	{
		if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, registryKey.getString(), 0, KEY_SET_VALUE, &netKey) == ERROR_SUCCESS)
		{
			if (RegDeleteValue(netKey, key.getString()) == ERROR_SUCCESS)
			{
				RegCloseKey(netKey);
				return true;
			}
			RegCloseKey(netKey);
			return false;
		}
		else
		{
			return false;
		}
	}
	if (registry == e_currentUser)
	{
		if(RegOpenKeyEx(HKEY_CURRENT_USER, registryKey.getString(), 0, KEY_SET_VALUE, &netKey) == ERROR_SUCCESS)
		{
			if (RegDeleteValue(netKey, key.getString()) == ERROR_SUCCESS)
			{
				RegCloseKey(netKey);
				return true;
			}
			RegCloseKey(netKey);
			return false;
		}
		else
		{
			return false;
		}
	}

	return false;
#else
	CSystemString theRegKey = registryKey;
	CSystemString theKey    = key;

	// Of the form com.exponent.detlaspace
	CFStringRef applicationID = theRegKey.getStringAsCFStringRef();
	CFStringRef cfKey     	  = theKey.getStringAsCFStringRef();
	bool complete			  = false;

	// Check that they are all valid
	if (applicationID && cfKey)
	{
		CFPreferencesSetAppValue(cfKey, NULL, applicationID);
		CFPreferencesAppSynchronize(applicationID);
		complete = true;
	}

	if (applicationID)
	{
		CFRelease(applicationID);
	}

	if (cfKey)
	{
		CFRelease(cfKey);
	}

	return complete;
#endif
}

//  ===========================================================================
TCountedPointerArray<CRegistryInfo> CRegistry::enumerateRegistryKeyValues(const ERegistry registry, const CString &registryKey)
{
	// Create the array that we will return
	TCountedPointerArray<CRegistryInfo> values;
#ifdef WIN32
	HKEY netKey;
	if (registry == e_localMachine)
	{
		if(RegOpenKeyEx(HKEY_LOCAL_MACHINE, registryKey.getString(), 0, KEY_QUERY_VALUE, &netKey) == ERROR_SUCCESS)
		{
			TCHAR key[16383];
			TCHAR value[16383];
			DWORD keySize	  = 16383;
			DWORD valueSize	  = 16383;
			DWORD returnValue = ERROR_SUCCESS;
			DWORD index		  = -1;

			do
			{
				index++;
				keySize		= 16383;
				valueSize	= 16383;
				key[0]		= '\0';
				value[0]	= '\0';
				returnValue = RegEnumValue(netKey, index, key, &keySize, NULL, NULL, (LPBYTE)value, &valueSize);
				if (returnValue == ERROR_SUCCESS)
				{
					values.addElement(new CRegistryInfo(key, value));
				}
			}
			while(returnValue != ERROR_NO_MORE_ITEMS);
			RegCloseKey(netKey);
		}
	}
	else if (registry == e_currentUser)
	{
		if(RegOpenKeyEx(HKEY_CURRENT_USER, registryKey.getString(), 0, KEY_QUERY_VALUE, &netKey) == ERROR_SUCCESS)
		{
			TCHAR key[16383];
			TCHAR value[16383];
			DWORD keySize	  = 16383;
			DWORD valueSize	  = 16383;
			DWORD returnValue = ERROR_SUCCESS;
			DWORD index		  = -1;

			do
			{
				index++;
				keySize		= 16383;
				valueSize	= 16383;
				key[0]		= '\0';
				value[0]	= '\0';
				returnValue = RegEnumValue(netKey, index, key, &keySize, NULL, NULL, (LPBYTE)value, &valueSize);
				if (returnValue == ERROR_SUCCESS)
				{
					values.addElement(new CRegistryInfo(key, value));
				}
			}
			while(returnValue != ERROR_NO_MORE_ITEMS);
			RegCloseKey(netKey);
		}
	}
#else
	CSystemString theRegKey = registryKey;

	// Of the form com.exponent.detlaspace
	CFStringRef applicationID = theRegKey.getStringAsCFStringRef();
	CFArrayRef array = NULL;

	if (applicationID)
	{
		array = CFPreferencesCopyKeyList(applicationID, kCFPreferencesCurrentUser, kCFPreferencesCurrentHost);
		CFRelease(applicationID);
	}

	if (array)
	{
		// Get the size of the array
		CFIndex count = CFArrayGetCount(array);

		// Loop through and get each value
		for (CFIndex i = 0; i < count; i++)
		{
			const CFStringRef value = (CFStringRef)CFArrayGetValueAtIndex(array, i);
			if (value)
			{
				// Adding it to the array if its valid
				CString key;
				key.setStringFromCFStringRef(value);
				values.addElement(new CRegistryInfo(key, CString::CSTRING_EMPTY_STRING));
			}
		}

		// Finally release the array
		CFRelease(array);
	}
#endif
	return values;
}